<?php
session_start();
require_once "./utils/config.php";
require_once "./dbcon.php";

// First, let's get the correct column name for email from usertable
$query = "SHOW COLUMNS FROM usertable";
$result = $con->query($query);
$emailColumn = '';
while ($row = $result->fetch_assoc()) {
    // Look for common email column names
    if (strtolower($row['Field']) === 'email' || 
        strtolower($row['Field']) === 'user_email' || 
        strtolower($row['Field']) === 'emailid') {
        $emailColumn = $row['Field'];
        break;
    }
}

if (empty($emailColumn)) {
    die("Error: Could not find email column in usertable. Please check your database structure.");
}

// Fetch all user details from usertable
$query = "SELECT $emailColumn, walletamount, name, mobile FROM usertable";
$result = $con->query($query);
$users = [];
while ($row = $result->fetch_assoc()) {
    $users[] = $row;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if(isset($_POST['action'])) {
        $email = $_POST['email'];
        $amount = floatval($_POST['amount']);

        if($_POST['action'] === 'update_balance') {
            $type = $_POST['type'];
            
            if ($type === 'credit') {
                $sql = "UPDATE usertable SET walletamount = walletamount + ? WHERE $emailColumn = ?";
            } else {
                $sql = "UPDATE usertable SET walletamount = walletamount - ? WHERE $emailColumn = ?";
            }

            try {
                $stmt = $con->prepare($sql);
                $stmt->bind_param("ds", $amount, $email);
                
                if ($stmt->execute()) {
                    $message = "Balance successfully " . ($type === 'credit' ? "added to" : "deducted from") . " wallet!";
                    $messageType = "success";
                } else {
                    $message = "Error updating balance!";
                    $messageType = "danger";
                }
                $stmt->close();
            } catch (Exception $e) {
                $message = "Error: " . $e->getMessage();
                $messageType = "danger";
            }
        } elseif($_POST['action'] === 'proceed_payment') {
            // Get user details for the selected email
            $selectedUser = null;
            foreach($users as $user) {
                if($user[$emailColumn] === $email) {
                    $selectedUser = $user;
                    break;
                }
            }

            if($selectedUser) {
                // Create a form and submit it to pay.php
                echo "
                <form id='paymentForm' action='pay.php' method='POST'>
                    <input type='hidden' name='name' value='" . htmlspecialchars($selectedUser['name']) . "'>
                    <input type='hidden' name='email' value='" . htmlspecialchars($email) . "'>
                    <input type='hidden' name='contact' value='" . htmlspecialchars($selectedUser['mobile']) . "'>
                    <input type='hidden' name='amount' value='" . htmlspecialchars($amount) . "'>
                </form>
                <script>document.getElementById('paymentForm').submit();</script>
                ";
                exit;
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Balance</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background: #f8f9fa;
            padding: 20px;
        }
        .card {
            max-width: 600px;
            margin: 50px auto;
            border-radius: 15px;
            box-shadow: 0 0 15px rgba(0,0,0,0.1);
        }
        .card-header {
            background: #2563eb;
            color: white;
            border-radius: 15px 15px 0 0 !important;
        }
        .form-select, .form-control {
            border-radius: 8px;
        }
        .btn-group {
            gap: 10px;
        }
        .current-balance {
            font-size: 1.1em;
            color: #0d6efd;
            font-weight: bold;
        }
        .user-details {
            margin-top: 10px;
            padding: 10px;
            background: #f8f9fa;
            border-radius: 8px;
            display: none;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <div class="card-header">
                <h4 class="mb-0">Manage Wallet Balance</h4>
            </div>
            <div class="card-body">
                <?php if (isset($message)): ?>
                    <div class="alert alert-<?php echo $messageType; ?>" role="alert">
                        <?php echo htmlspecialchars($message); ?>
                    </div>
                <?php endif; ?>

                <form method="POST" action="" id="balanceForm">
                    <div class="mb-3">
                        <label for="email" class="form-label">Select User Email</label>
                        <select name="email" id="email" class="form-select" required onchange="updateUserDetails()">
                            <option value="">Select Email</option>
                            <?php foreach ($users as $user): ?>
                                <option value="<?php echo htmlspecialchars($user[$emailColumn]); ?>" 
                                        data-balance="<?php echo htmlspecialchars($user['walletamount']); ?>"
                                        data-name="<?php echo htmlspecialchars($user['name']); ?>"
                                        data-mobile="<?php echo htmlspecialchars($user['mobile']); ?>">
                                    <?php echo htmlspecialchars($user[$emailColumn]); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="user-details mb-3" id="userDetails">
                        <p><strong>Name:</strong> <span id="userName"></span></p>
                        <p><strong>Mobile:</strong> <span id="userMobile"></span></p>
                        <p><strong>Current Balance:</strong> <span id="currentBalance" class="current-balance">0.00</span></p>
                    </div>

                    <div class="mb-3">
                        <label for="type" class="form-label">Transaction Type</label>
                        <select name="type" id="type" class="form-select" required>
                            <option value="credit">Credit (Add Money)</option>
                            <option value="debit">Debit (Subtract Money)</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label for="amount" class="form-label">Amount</label>
                        <input type="number" class="form-control" id="amount" name="amount" 
                               step="0.01" min="0" required>
                    </div>

                    <div class="d-flex justify-content-between">
                        <button type="submit" name="action" value="update_balance" 
                                class="btn btn-primary">Update Balance</button>
                        <button type="submit" name="action" value="proceed_payment" 
                                class="btn btn-success">Proceed to Payment</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function updateUserDetails() {
            const select = document.getElementById('email');
            const selectedOption = select.options[select.selectedIndex];
            const userDetails = document.getElementById('userDetails');
            
            if (selectedOption.value) {
                document.getElementById('userName').textContent = selectedOption.getAttribute('data-name');
                document.getElementById('userMobile').textContent = selectedOption.getAttribute('data-mobile');
                document.getElementById('currentBalance').textContent = 
                    parseFloat(selectedOption.getAttribute('data-balance')).toFixed(2);
                userDetails.style.display = 'block';
            } else {
                userDetails.style.display = 'none';
            }
        }

        document.getElementById('balanceForm').addEventListener('submit', function(e) {
            const amount = parseFloat(document.getElementById('amount').value);
            const type = document.getElementById('type').value;
            const currentBalance = parseFloat(document.getElementById('currentBalance').textContent);

            if (type === 'debit' && amount > currentBalance) {
                e.preventDefault();
                alert('Insufficient balance! Cannot debit more than current balance.');
            }
        });
    </script>
</body>
</html> 