<?php
session_start();
require_once "./utils/config.php";
require_once "./utils/common.php";
require_once "./dbcon.php";

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// Get user data
$user_id = $_SESSION['user_id'];
$user_name = $_SESSION['user_name'];

// Get current balance
$balance = getUserBalance($user_id, $con);

// Get recent transactions (last 5)
$stmt = $con->prepare("
    SELECT 
        transaction_id,
        amount,
        type,
        status,
        created_at
    FROM balance_transactions 
    WHERE user_id = ?
    ORDER BY created_at DESC
    LIMIT 5
");
$stmt->bind_param("s", $user_id);
$stmt->execute();
$transactions = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);

// Get payment history (last 5)
$stmt = $con->prepare("
    SELECT 
        transaction_id,
        amount,
        status,
        created_at
    FROM payment_attempts 
    WHERE email = ?
    ORDER BY created_at DESC
    LIMIT 5
");
$stmt->bind_param("s", $_SESSION['email']);
$stmt->execute();
$payments = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.8.1/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        .dashboard-container {
            padding: 30px 0;
        }
        .balance-card {
            background: linear-gradient(45deg, #0d6efd, #0dcaf0);
            color: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
        }
        .balance-amount {
            font-size: 2.5rem;
            font-weight: bold;
        }
        .transaction-card {
            border-radius: 10px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        .transaction-item {
            padding: 15px;
            border-bottom: 1px solid #eee;
        }
        .transaction-item:last-child {
            border-bottom: none;
        }
        .status-badge {
            font-size: 0.8rem;
            padding: 5px 10px;
            border-radius: 15px;
        }
        .logout-btn {
            position: absolute;
            top: 20px;
            right: 20px;
        }
    </style>
</head>
<body class="bg-light">
    <div class="container dashboard-container">
        <a href="logout.php" class="btn btn-outline-danger logout-btn">
            <i class="bi bi-box-arrow-right"></i> Logout
        </a>

        <div class="row">
            <div class="col-md-12 mb-4">
                <h2>Welcome, <?php echo htmlspecialchars($user_name); ?>!</h2>
            </div>
        </div>

        <div class="row">
            <!-- Wallet Balance Section -->
            <div class="col-md-4">
                <div class="balance-card">
                    <h4>Wallet Balance</h4>
                    <div class="balance-amount">₹<?php echo number_format($balance, 2); ?></div>
                    <div class="mt-3">
                        <a href="add_balance.php" class="btn btn-light">
                            <i class="bi bi-plus-circle"></i> Add Money
                        </a>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="col-md-8">
                <div class="card transaction-card">
                    <div class="card-header">
                        <h5 class="mb-0">Quick Actions</h5>
                    </div>
                    <div class="card-body">
                        <div class="row">
                            <div class="col-6 col-md-4 mb-3">
                                <a href="make_payment.php" class="btn btn-primary w-100">
                                    <i class="bi bi-credit-card"></i> Make Payment
                                </a>
                            </div>
                            <div class="col-6 col-md-4 mb-3">
                                <a href="transaction_history.php" class="btn btn-info w-100">
                                    <i class="bi bi-clock-history"></i> Transaction History
                                </a>
                            </div>
                            <div class="col-6 col-md-4 mb-3">
                                <a href="profile.php" class="btn btn-secondary w-100">
                                    <i class="bi bi-person"></i> Profile
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row mt-4">
            <!-- Recent Wallet Transactions -->
            <div class="col-md-6">
                <div class="card transaction-card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Recent Wallet Transactions</h5>
                        <a href="transaction_history.php" class="btn btn-sm btn-outline-primary">View All</a>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($transactions)): ?>
                            <div class="p-3 text-center text-muted">
                                No recent transactions
                            </div>
                        <?php else: ?>
                            <?php foreach ($transactions as $transaction): ?>
                                <div class="transaction-item d-flex justify-content-between align-items-center">
                                    <div>
                                        <div class="fw-bold">
                                            <?php echo $transaction['type'] === 'credit' ? 'Added' : 'Spent'; ?>
                                            ₹<?php echo number_format($transaction['amount'], 2); ?>
                                        </div>
                                        <small class="text-muted">
                                            <?php echo date('d M Y, h:i A', strtotime($transaction['created_at'])); ?>
                                        </small>
                                    </div>
                                    <div>
                                        <span class="status-badge bg-<?php echo $transaction['status'] === 'COMPLETED' ? 'success' : 'warning'; ?>">
                                            <?php echo $transaction['status']; ?>
                                        </span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Recent Payments -->
            <div class="col-md-6">
                <div class="card transaction-card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Recent Payments</h5>
                        <a href="payment_history.php" class="btn btn-sm btn-outline-primary">View All</a>
                    </div>
                    <div class="card-body p-0">
                        <?php if (empty($payments)): ?>
                            <div class="p-3 text-center text-muted">
                                No recent payments
                            </div>
                        <?php else: ?>
                            <?php foreach ($payments as $payment): ?>
                                <div class="transaction-item d-flex justify-content-between align-items-center">
                                    <div>
                                        <div class="fw-bold">
                                            ₹<?php echo number_format($payment['amount'], 2); ?>
                                        </div>
                                        <small class="text-muted">
                                            <?php echo date('d M Y, h:i A', strtotime($payment['created_at'])); ?>
                                        </small>
                                    </div>
                                    <div>
                                        <span class="status-badge bg-<?php echo $payment['status'] === 'PAYMENT_SUCCESS' ? 'success' : 'warning'; ?>">
                                            <?php echo $payment['status']; ?>
                                        </span>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 