<?php
session_start();
require_once "./utils/config.php";
require_once "./utils/common.php";
require_once "./dbcon.php";

// Check if payment data exists in session
if (!isset($_SESSION['payment_data'])) {
    header("Location: error.php?message=invalid_payment_request");
    exit;
}

$paymentData = $_SESSION['payment_data'];

// Validate required data
if (!isset($paymentData['amount'], $paymentData['transaction_id'], $paymentData['merchant_id'])) {
    error_log("Missing required payment data");
    header("Location: error.php?message=invalid_payment_data");
    exit;
}

try {
    // Prepare payment payload
    $payLoad = [
        'merchantId' => $paymentData['merchant_id'],
        'merchantTransactionId' => $paymentData['transaction_id'],
        'merchantUserId' => isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 'GUEST-' . time(),
        'amount' => $paymentData['amount'] * 100, // Convert to paisa
        'redirectUrl' => BASE_URL . REDIRECTURL,
        'redirectMode' => "POST",
        'callbackUrl' => BASE_URL . REDIRECTURL,
        'mobileNumber' => $paymentData['contact'] ?? '',
        'paymentInstrument' => [
            'type' => "PAY_PAGE"
        ]
    ];

    // Add optional parameters if available
    if (isset($paymentData['email']) && !empty($paymentData['email'])) {
        $payLoad['email'] = $paymentData['email'];
    }

    // Convert payload to JSON and encode
    $jsonEncodedPayload = json_encode($payLoad);
    $payloadBase64 = base64_encode($jsonEncodedPayload);

    // Generate checksum
    $saltKey = (API_STATUS === "LIVE") ? SALTKEYLIVE : SALTKEYUAT;
    $saltIndex = SALTINDEX;
    
    // Create checksum data
    $dataToHash = $payloadBase64 . "/pg/v1/pay" . $saltKey;
    $checksum = hash("sha256", $dataToHash) . "###" . $saltIndex;

    // Log payment request details
    error_log("Payment Gateway Request - Transaction ID: {$paymentData['transaction_id']}");
    error_log("Payload: " . print_r($payLoad, true));
    error_log("Checksum: " . $checksum);

    // Determine API URL based on environment
    $url = (API_STATUS === "LIVE") ? LIVEURLPAY : UATURLPAY;

    // Prepare cURL request
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode([
            'request' => $payloadBase64
        ]),
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'X-VERIFY: ' . $checksum,
        ],
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false
    ]);

    // Execute cURL request
    $response = curl_exec($curl);
    $err = curl_error($curl);
    
    curl_close($curl);

    if ($err) {
        throw new Exception("cURL Error: " . $err);
    }

    // Decode response
    $responseData = json_decode($response, true);

    // Log response
    error_log("Payment Gateway Response: " . print_r($responseData, true));

    // Check response
    if (!isset($responseData['success']) || !$responseData['success']) {
        throw new Exception("Payment gateway error: " . ($responseData['message'] ?? 'Unknown error'));
    }

    // Store payment attempt in database
    $stmt = $con->prepare("
        INSERT INTO payment_attempts (
            transaction_id, merchant_id, email, amount, status, 
            name, contact, created_at, updated_at
        ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
    ");
    
    $initialStatus = "PAYMENT_INITIATED";
    $stmt->bind_param("sssdsss", 
        $paymentData['transaction_id'],
        $paymentData['merchant_id'],
        $paymentData['email'],
        $paymentData['amount'],
        $initialStatus,
        $paymentData['name'],
        $paymentData['contact']
    );
    
    if (!$stmt->execute()) {
        throw new Exception("Failed to store payment attempt");
    }

    // Redirect to payment page
    if (isset($responseData['data']['instrumentResponse']['redirectInfo']['url'])) {
        header("Location: " . $responseData['data']['instrumentResponse']['redirectInfo']['url']);
        exit;
    } else {
        throw new Exception("No redirect URL in response");
    }

} catch (Exception $e) {
    error_log("Payment Gateway Error: " . $e->getMessage());
    header("Location: error.php?message=payment_gateway_error&transaction_id=" . 
           ($paymentData['transaction_id'] ?? 'unknown'));
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Processing Payment</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .loading-container {
            height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            flex-direction: column;
        }
        .spinner-border {
            width: 4rem;
            height: 4rem;
        }
    </style>
</head>
<body>
    <div class="loading-container">
        <div class="spinner-border text-primary mb-3" role="status">
            <span class="visually-hidden">Loading...</span>
        </div>
        <h3>Processing your payment...</h3>
        <p class="text-muted">Please do not refresh or close this page.</p>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html> 