<?php
session_start();
require_once "./utils/config.php";
require_once "./utils/common.php";
require_once "./dbcon.php";

// Initialize variables
$transactionId = $_POST['transactionId'] ?? $_GET['transactionId'] ?? null;
$merchantId = $_POST['merchantId'] ?? $_GET['merchantId'] ?? null;
$status = $_POST['code'] ?? $_GET['code'] ?? 'UNKNOWN';

// Validate request
if (!$transactionId) {
    error_log("Payment Status: Missing transaction ID");
    header("Location: error.php?message=invalid_transaction");
    exit;
}

try {
    // Check existing payment status from balance_transactions
    $stmt = $con->prepare("
        SELECT bt.status, bt.amount, bt.user_id, u.email, u.name 
        FROM balance_transactions bt
        LEFT JOIN users u ON bt.user_id = u.id
        WHERE bt.transaction_id = ?
    ");
    $stmt->bind_param("s", $transactionId);
    $stmt->execute();
    $result = $stmt->get_result();
    $paymentData = $result->fetch_assoc();

    if (!$paymentData) {
        throw new Exception("Transaction record not found");
    }

    // API configuration for status check
    $url = LIVESTATUSCHECKURL . $merchantId . "/" . $transactionId;
    $saltKey = SALTKEYLIVE;
    $saltIndex = SALTINDEX;

    // Create checksum for API request
    $dataToHash = "/pg/v1/status/" . $merchantId . "/" . $transactionId . $saltKey;
    $checksum = hash("sha256", $dataToHash) . "###" . $saltIndex;

    // Set up cURL request
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => [
            "Content-Type: application/json",
            "X-VERIFY: " . $checksum,
            "X-MERCHANT-ID: " . $merchantId
        ],
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_SSL_VERIFYPEER => 0
    ]);

    $response = curl_exec($curl);
    $error = curl_error($curl);
    curl_close($curl);

    if ($error) {
        throw new Exception("Status check failed: " . $error);
    }

    // Process response
    $responseData = json_decode($response, true);
    $finalStatus = $responseData['code'] ?? 'UNKNOWN';

    // Begin transaction for status update
    $con->begin_transaction();

    // Update balance_transactions status
    $stmt = $con->prepare("
        UPDATE balance_transactions 
        SET status = ?, 
            updated_at = NOW() 
        WHERE transaction_id = ?
    ");
    $stmt->bind_param("ss", $finalStatus, $transactionId);
    $stmt->execute();

    // If payment is successful, update user balance
    if ($finalStatus === 'PAYMENT_SUCCESS') {
        $stmt = $con->prepare("
            UPDATE usertable 
            SET walletamount = walletamount + ?
            WHERE userid = ?
        ");
        $stmt->bind_param("ds", 
            $paymentData['amount'],
            $paymentData['user_id']
        );
        $stmt->execute();

        if ($stmt->affected_rows === 0) {
            throw new Exception("Failed to update wallet balance");
        }
    }

    $con->commit();

    // Clear session payment data
    unset($_SESSION['payment_data']);

    // Redirect based on status
    if ($finalStatus === 'PAYMENT_SUCCESS') {
        header("Location: success.php?tid=" . $transactionId);
    } else {
        header("Location: failure.php?tid=" . $transactionId);
    }
    exit;

} catch (Exception $e) {
    $con->rollback();
    error_log("Payment status error: " . $e->getMessage());
    header("Location: error.php?message=payment_status_failed&tid=" . $transactionId);
    exit;
}

// Helper function for status messages
function getStatusMessage($status) {
    $messages = [
        'PAYMENT_SUCCESS' => 'Payment completed successfully',
        'PAYMENT_ERROR' => 'Payment failed',
        'PAYMENT_PENDING' => 'Payment is being processed',
        'PAYMENT_INITIATED' => 'Payment has been initiated',
        'PAYMENT_CANCELLED' => 'Payment was cancelled',
        'UNKNOWN' => 'Payment status unknown'
    ];
    return $messages[$status] ?? $messages['UNKNOWN'];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Status</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        .status-container {
            max-width: 600px;
            margin: 50px auto;
            padding: 20px;
        }
        .status-card {
            border-radius: 15px;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
        }
        .status-header {
            padding: 20px;
            border-radius: 15px 15px 0 0;
            color: white;
        }
        .status-success { background-color: #198754; }
        .status-pending { background-color: #ffc107; }
        .status-error { background-color: #dc3545; }
        .status-body {
            padding: 30px;
        }
    </style>
</head>
<body class="bg-light">
    <?php if (!empty($_SERVER['HTTP_X_REQUESTED_WITH'])): ?>
        <div class="status-container">
            <div class="status-card">
                <div class="status-header <?php echo getStatusClass($finalStatus); ?>">
                    <h3 class="mb-0"><?php echo getStatusMessage($finalStatus); ?></h3>
                </div>
                <div class="status-body">
                    <p><strong>Transaction ID:</strong> <?php echo htmlspecialchars($transactionId); ?></p>
                    <p><strong>Amount:</strong> ₹<?php echo number_format($paymentData['amount'], 2); ?></p>
                    <p><strong>Status:</strong> <?php echo htmlspecialchars($finalStatus); ?></p>
                    
                    <div class="mt-4">
                        <?php if ($finalStatus === 'PAYMENT_SUCCESS'): ?>
                            <a href="dashboard.php" class="btn btn-primary">Back to Dashboard</a>
                        <?php else: ?>
                            <a href="add_balance.php" class="btn btn-primary">Try Again</a>
                            <a href="dashboard.php" class="btn btn-secondary">Back to Dashboard</a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    <?php endif; ?>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>

<?php
// Helper function for status classes
function getStatusClass($status) {
    switch ($status) {
        case 'PAYMENT_SUCCESS':
            return 'status-success';
        case 'PAYMENT_PENDING':
        case 'PAYMENT_INITIATED':
            return 'status-pending';
        default:
            return 'status-error';
    }
}
?>